/*
 * Copyright 2020 Renjie Wu <rwu034 AT ucr.edu>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and0
 * limitations under the License.
 */

const MatlabCTreeLexer = require('../generated/MatlabCTreeLexer').MatlabCTreeLexer;
const MatlabCTreeVisitor = require('../generated/MatlabCTreeVisitor').MatlabCTreeVisitor;

class ASTVisitor extends MatlabCTreeVisitor {
    constructor(jsonNodes) {
        super();

        this.complementOperators = {
            "<": ">=", ">": "<=",
            "<=": ">", ">=": "<",
            "==": "~=", "~=": "=="
        };

        this.nodeNumber = 0;
        this.errors = [];

        this.jsonNodes = jsonNodes;
    }

    visitStmt(ctx) {
        // stmt: INTEGER (ifStmt | classExpr);
        let nodeNumber = parseInt(ctx.children[0].getText());

        // Check if the counter matches node number in stmt
        if (++this.nodeNumber !== nodeNumber) {
            let symbol = ctx.children[0].symbol;
            this.errors.push({
                line: symbol.line,
                message: `mismatched node number ${nodeNumber} expecting ${this.nodeNumber}`
            });
        }

        let result = this.visitChildren(ctx);
        this.jsonNodes.push(result[1]);

        // Actually it doesn't matter whether result is returned
        return result;
    }

    visitClassExpr(ctx) {
        // classExpr: 'class' '=' labelExpr;
        return ctx.children[2].getText();
    }

    visitIfStmt(ctx) {
        // ifStmt: ifClause elseifClause? elseClause?;
        let result = this.visitChildren(ctx);

        let featureName = result[0].condition.identifer;
        let operator = result[0].condition.operator;
        let compareValue = result[0].condition.value;
        let trueBranch = result[0].branch;
        let falseBranch = null;

        // 2 or more clauses?
        if (result.length > 1) {
            // elseifClause
            if (result[1].hasOwnProperty("condition")) {
                let elseifFN = result[1].condition.identifer;
                let elseifOP = result[1].condition.operator;
                let elseifCV = result[1].condition.value;

                // Does conditions match?
                if (elseifFN !== featureName ||
                    elseifOP !== this.complementOperators[operator] ||
                    elseifCV !== compareValue) {

                    let elseIfExpr = ctx.children[1].children[1];
                    this.errors.push({
                        line: elseIfExpr.start.line,
                        message: `mismatched elseif clause '${elseIfExpr.getText()}' ` +
                            `expecting '${featureName}${this.complementOperators[operator]}${compareValue}'`
                    });
                } else {
                    falseBranch = result[1].branch;
                }
            } else {
                // elseClause
                falseBranch = result[1];
            }
        }

        // We ignored the elseClause here if there are 3 clauses present.
        // This should be safe since the generated classification tree from
        // MATLAB is supposed to be a binary tree.

        return { featureName, operator, compareValue, trueBranch, falseBranch };
    }

    visitSelectClause(ctx) {
        // ifClause    : 'if' condExpr 'then' nodeExpr;
        // elseifClause: 'elseif' condExpr 'then' nodeExpr;
        let result = this.visitChildren(ctx);

        return { condition: result[1], branch: result[3] };
    }

    visitIfClause(ctx) {
        return this.visitSelectClause(ctx);
    }

    visitElseifClause(ctx) {
        return this.visitSelectClause(ctx);
    }

    visitElseClause(ctx) {
        // elseClause: 'else' nodeExpr;
        return this.visitChildren(ctx)[1];
    }

    visitConstExpr(ctx) {
        // constExpr: INTEGER | RATIONAL;
        let number = ctx.children[0];

        if (number.symbol.type === MatlabCTreeLexer.INTEGER) {
            return parseInt(number.getText());
        } else { // RATIONAL
            return parseFloat(number.getText());
        }
    }

    visitCondExpr(ctx) {
        // condExpr: IDENTIFIER op constExpr;
        let result = this.visitChildren(ctx);

        return { identifer: ctx.children[0].getText(), operator: ctx.children[1].getText(), value: result[2] };
    }

    visitNodeExpr(ctx) {
        // nodeExpr: 'node' INTEGER | labelExpr;
        if (ctx.children.length > 1) {
            // 'node' INTEGER
            let targetNode = parseInt(ctx.children[1].getText());
            return { type: "node", target: targetNode, line: ctx.children[1].symbol.line };
        } else {
            // labelExpr
            let label = ctx.children[0].getText();
            return { type: "label", label: label };
        }
    }

    getErrors() {
        return this.errors;
    }
}

module.exports.ASTVisitor = ASTVisitor;