/*
 * Copyright 2020 Renjie Wu <rwu034 AT ucr.edu>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and0
 * limitations under the License.
 */

class JSONGenerator {
    constructor(jsonNodes, compactOutput) {
        this.errors = [];
        this.visitedNodes = [];

        this.jsonNodes = jsonNodes;

        this.outputFN = compactOutput ? "fn" : "featureName";
        this.outputOP = compactOutput ? "op" : "operator";
        this.outputCV = compactOutput ? "val" : "compareValue";
        this.outputTB = compactOutput ? "tb" : "trueBranch";
        this.outputFB = compactOutput ? "fb" : "falseBranch";
    }

    visitBranch(branch) {
        // False branch could be null
        if (!branch) {
            return null;
        }

        if (branch.type === "label") {
            return branch.label;
        } else { // "node"
            if (branch.target > this.jsonNodes.length) {
                this.errors.push({
                    line: branch.line,
                    message: `invalid node index ${branch.target} ` +
                        `expecting no greater than ${this.jsonNodes.length}`
                });
                return null;
            } else {
                // Count starting from 1 in MATLAB output
                let targetIndex = branch.target - 1;
                let targetNode = this.jsonNodes[targetIndex];

                // classExpr node
                if (typeof targetNode === "string") {
                    this.visitedNodes.push(targetIndex);
                    return targetNode;
                } else if (!this.visitedNodes.includes(targetIndex)) {
                    // ifStmt node. Visit if haven't
                    return this.dfs(targetNode, targetIndex);
                }
            }
        }
    }

    dfs(node, index) {
        let result = {};

        this.visitedNodes.push(index);

        result[this.outputFN] = node.featureName;
        result[this.outputOP] = node.operator;
        result[this.outputCV] = node.compareValue;
        result[this.outputTB] = this.visitBranch(node.trueBranch);
        result[this.outputFB] = this.visitBranch(node.falseBranch);

        return result;
    }

    generate() {
        let jsonOutput = [];

        // Perform DFS on all nodes so we can also handle forests
        this.jsonNodes.forEach((node, index) => {
            // Only take care not yet visited ifStmt nodes
            if (typeof node !== "string" && !this.visitedNodes.includes(index)) {
                jsonOutput.push(this.dfs(node, index));
            }
        });

        this.jsonNodes.forEach((node, index) => {
            // Boundary condition, the tree only has a label.
            if (typeof node === "string" && !this.visitedNodes.includes(index)) {
                jsonOutput.push(node);
            }
        });

        return jsonOutput;
    }

    getErrors() {
        return this.errors;
    }
}

module.exports.JSONGenerator = JSONGenerator;