/*
 * Copyright 2020 Renjie Wu <rwu034 AT ucr.edu>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and0
 * limitations under the License.
 */

const antlr4 = require('antlr4');
const MatlabCTreeLexer = require('../generated/MatlabCTreeLexer').MatlabCTreeLexer;
const MatlabCTreeParser = require('../generated/MatlabCTreeParser').MatlabCTreeParser;
const ASTVisitor = require('./ASTVisitor').ASTVisitor;
const JSONGenerator = require('./JSONGenerator').JSONGenerator;

class MatlabCTreeConverter {
    toJson(inputCTree, compactOutput) {
        compactOutput = !!compactOutput || false;
    
        let chars = new antlr4.InputStream(inputCTree);
        let lexer = new MatlabCTreeLexer(chars);
        let tokens = new antlr4.CommonTokenStream(lexer);
        let parser = new MatlabCTreeParser(tokens);
    
        let syntaxErrors = [];
        let jsonNodes = [];
    
        // Adapted from https://stackoverflow.com/questions/30276048/handling-errors-in-antlr4-javascript
        parser.removeErrorListeners();
        parser.addErrorListener({
            syntaxError: (recognizer, offendingSymbol, line, column, msg, err) => {
                syntaxErrors.push({ line: line, message: msg });
            }
        })
        
        parser.buildParseTrees = true;
        let ast = parser.start();
    
        if (syntaxErrors.length > 0) {
            return { json: null, errors: syntaxErrors };
        }
    
        let astVisitor = new ASTVisitor(jsonNodes);
        ast.accept(astVisitor);
    
        if (astVisitor.getErrors().length > 0) {
            return { json: null, errors: astVisitor.getErrors() };
        }
    
        let jsonGenerator = new JSONGenerator(jsonNodes, compactOutput);
        let outputJson = jsonGenerator.generate();
    
        if (jsonGenerator.getErrors().length > 0) {
            return { json: null, errors: jsonGenerator.getErrors() };
        }
    
        return { json: outputJson, errors: null };
    }
}

module.exports.MatlabCTreeConverter = MatlabCTreeConverter;